#ifndef MERGEREDUCE_HPP
#define	MERGEREDUCE_HPP

#include <vector>
#include <memory>

/**
 * @brief Merge & Reduce framework template
 * 
 * Can be used to form an offline algorithm into a streaming algorithm
 */
template<typename T> class MergeReduce
{
private:
    std::vector<std::unique_ptr<std::vector<T >> > buckets;
    int firstBucketSize;

public:

    /**
     * @param firstBucketSize Size of initial Merge & Reduce bucket
     */
    MergeReduce(int firstBucketSize) :
    buckets(),
    firstBucketSize(firstBucketSize)
    {
        buckets.push_back(std::unique_ptr < std::vector < T >> (new std::vector<T>()));
        buckets.reserve(firstBucketSize);
    }

    virtual ~MergeReduce()
    {
    }

    /**
     * Streaming operator for reading points
     * @param element Point
     * @return This object
     */
    MergeReduce& operator<<(T const & element);
    virtual std::unique_ptr<std::vector<T >> assemble();

private:
    /**
     * @brief Private insertion method
     * @param element Point
     */
    virtual void insert(T const & element);
    
    /**
     * @brief Private main method for reducing
     */
    virtual void reduce();
    
    /**
     * @brief Private recursive method for reducing
     * @param input Point set to be reduce
     * @param level Level of the point set
     * @param reduced Output iterator for reduced set
     */
    virtual void reduce(std::vector<T> const * input, int level, std::vector<T> * reduced) = 0;
    
    /**
     * @brief Private method for merging
     * @param left Left child
     * @param right Right child
     * @param level Level of children
     * @param merged Output iterator for merged set
     */
    virtual void merge(std::vector<T> const * left, std::vector<T> const * right, int level, std::vector<T> * merged);
};

template<typename T> MergeReduce<T>& MergeReduce<T>::operator <<(T const & element)
{
    if (buckets[0]->size() >= firstBucketSize)
        reduce();
    insert(element);

    return *this;
}

template<typename T> void MergeReduce<T>::reduce()
{
    int level = 0;
    std::unique_ptr < std::vector < T >> reduced(new std::vector<T>());
    std::unique_ptr < std::vector < T >> merged(new std::vector<T>());

    // Nothing to reduce
    if(buckets[level]->size() == 0)
        return;
    
    reduce(buckets[level].get(), level, reduced.get());
    buckets[level]->clear();
    ++level;
    while (buckets.size() >= level + 1 && buckets[level]->size() > 0)
    {
        merged->clear();
        merge(reduced.get(), buckets[level].get(), level, merged.get());
        buckets[level]->clear();
        reduced->clear();
        reduce(merged.get(), level, reduced.get());
        ++level;
    }
    if (buckets.size() < level + 1)
        buckets.push_back(std::unique_ptr < std::vector < T >> (new std::vector<T>()));
    buckets[level] = std::move(reduced);
}

template<typename T> std::unique_ptr<std::vector<T >> MergeReduce<T>::assemble()
{
    reduce();
    
    size_t num = 0;
    for (size_t i = 0; i < buckets.size(); ++i)
        num += buckets[i]->size();

    std::unique_ptr < std::vector < T >> assembly(new std::vector<T>());
    assembly->reserve(num);

    for (size_t i = 0; i < buckets.size(); ++i)
        for (size_t j = 0; j < buckets[i]->size(); ++j)
            assembly->push_back((*buckets[i])[j]);

    return assembly;
}

template<typename T> void MergeReduce<T>::insert(T const & element)
{
    buckets[0]->push_back(element);
}

template<typename T> void MergeReduce<T>::merge(std::vector<T> const * left, std::vector<T> const * right, int level, std::vector<T> * merged)
{
    merged->insert(merged->end(), left->begin(), left->end());
    merged->insert(merged->end(), right->begin(), right->end());
}

#endif	/* MERGEREDUCE_HPP */

